/*
 * Copyright (c) 2012, 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/* Set of constants and values used in RoundingAndPropertyTest.java.
 *
 * There are 5 different information in this class:
 * - TestLocale is the locale used by RoundingAndPropertyTest regression test
 *   when calling DecimalFormat.format() on either the set of DecimalGoldenValues
 *   or CurrencyGoldenValues.
 *   See main method of RoundingAndPropertyTest.
 *
 * - FullLocalizationTestLocale is the locale used by RoundingAndPropertyTest
 *   regression test when calling DecimalFormat.format() on the set of values
 *   (DecimalLocalizationValues) used to test that localization of digits
 *   happens correctly when needed.
 *   See main method of RoundingAndPropertyTest.
 *
 * - DecimalLocalizationValues is an array containing all the double values used
 *   to check that localization of digits happens correctly when needed.
 *   See RoundingAndPropertyTest.testLocalizationValues() method.
 *
 * - DecimalGoldenValues and CurrencyGoldenValues are arrays containing all the
 *   double values that will be used as input when checking correctness of
 *   results returned by DecimalFormat.format().
 *   2 arrays are needed since decimal and currency formatting patterns require
 *   a different number of digits after decimal point.
 *   See RoundingAndPropertyTest.testGoldenValues() method.
 *
 * - PROPERTY_CHECK_POSITIVE_VALUE and PROPERTY_CHECK_NEGATIVE_VALUE are the
 *   double values used for testing the validity of the property changes call
 *   in the fast-path case. The locale used in that case is TestLocale.
 *   See RoundingAndPropertyTest.testSettersAndFastPath() method.
 */

import java.util.*;

class GoldenDoubleValues {

    // TestLocale is the testing locale used by RoundingAndPropertyTest test,
    // when testing the golden double values
    static final Locale TestLocale = Locale.US;


    // FullTestLocale is the testing locale used by RoundingAndPropertyTest test,
    // when testing full localization of double values.
    static final Locale FullLocalizationTestLocale = Locale.of("hi", "IN");


    /* Below are the two double values used for exercising the changes of
     * of DecimalFormat properties and symbols. These values are also used
     * as golden values (see golden arrays below).
     */

    /* PROPERTY_CHECK_NEGATIVE_VALUE is the negative double value used for
     * testing the validity of the property changes for fast-path.
     * See testSettersAndFastPath() in RoundingAndPropertyTest test.
     */
    static final double PROPERTY_CHECK_NEGATIVE_VALUE = -2147483646.2334997d;

    /* PROPERTY_CHECK_POSITIVE_VALUE is the positive double value used for
     * testing the validity of the property changes for fast-path.
     * See testSettersAndFastPath() in RoundingAndPropertyTest test.
     */
    static final double PROPERTY_CHECK_POSITIVE_VALUE =  2147483646.2335003d;

    /* --- Array of double values to test localization ------------------------
     *
     * For most locales, effective localization does not happen on digits, i.e.
     * the digits are not changed due to localization. In order to check that
     * fast-path localize correctly digits in such a case, the array of double
     * values below deals with all the case of localization that may happen on
     * digits
     */
    static final double[] DecimalLocalizationValues = {
        1.123,
        12.123,
        123.123,
        1234.123,
        12345.123,
        123456.123,
        1234567.123,
        12345678.123,
        123456789.123,
        1234567890.123,
        1234.0,
        1234.9,
        1234.99,
        1234.999
    };


    /* --- Arrays of golden double values ----------------------------------
     *
     * The   GoldenValues arrays  are used   as input values   for checking the
     * correctness  of  the    DecimalFormat.format()  call  results  done   in
     * RoundingAndPropertyTest regression test. The results are compared to the
     * expected ones  found in  GoldenFormattedValues.  For each value   in the
     * arrays  there  is   a corresponding    expected  string  result   found,
     * represented as an array  of unicode values, at  the same index  in the
     * related GoldenFormattedValues array.  The string returned by  the format
     * call and the found in GoldenFormattedValues array  must be equal for the
     * result to be considered valid.
     * See RoundingAndPropertyTest.testGoldenValues() method.
     *
     * We  need 2  such  GoldenValues  arrays  since the decimal  and  currency
     * formatting rules require different number of digits after decimal point.
     *
     * Thus we have two different arrays of golden values:
     *  - DecimalGoldenValues for the decimal case.
     *  - CurrencyGoldenValues for the currency case.
     *
     * They are associated to related GoldenFormattedValues arrays, generated by
     * running RoundingAndPropertyTest with a "gengold" argument:
     *  - DecimalGoldenFormattedValues for the decimal case.
     *  - CurrencyGoldenFormattedValues for the currency case.
     * These two generated arrays are found in GoldenFormattedValues.java file.
     *
     * The  impact of the formatting rules  is as follows,  because the pattern
     * rule for  the  fractional  part is different   for  decimal and currency
     * patterns:
     *  - in decimal case one must output the first non-zero 3 digits of
     *    fractional part 1.1232 => "1.123" and 1.12016789 => "1.12"
     *  - in currency case the first 2 fractional digits are always output
     *    1.1232 => "1.12" and 1.0016789 => "1.00"
     *
     * Thus we  need a different  number of  fractional digits when  we specify
     * below the golden double values to check, and most of the decimal and
     * currency golden values differ only in the number of fractional digits.
     *
     * The list below exercises almost all code pathes of the fast-path
     * algorithm except localization of digits.
     */

    // --- The set of golden values for the Decimal formatting case --------
    static final double[] DecimalGoldenValues = {
        // Testing of specific values
        +0.0,
        -0.0,
        Double.MIN_VALUE,
        Double.MIN_NORMAL,
        PROPERTY_CHECK_NEGATIVE_VALUE,
        PROPERTY_CHECK_POSITIVE_VALUE,
        -2147483647.9996,
        2147483647.9996,
        -1999999999.9994997,
        1999999999.9995003,
        // Testing fast-path range checks (all outside bounds)
        Double.NaN,
        Double.POSITIVE_INFINITY,
        Double.NEGATIVE_INFINITY,
        Double.MAX_VALUE,
        -9876543210.9876543,
        9876543210.9876543,
        -1234567890.1234567E128,
        1234567890.1234567E128,
        // Testing of integral string size
        1.123,
        12.123,
        123.123,
        1234.123,
        12345.123,
        123456.123,
        1234567.123,
        12345678.123,
        123456789.123,
        1234567890.123,
        -1.123,
        -12.123,
        -123.123,
        -1234.123,
        -12345.123,
        -123456.123,
        -1234567.123,
        -12345678.123,
        -123456789.123,
        -1234567890.123,
        // Testing of fractional string size
        0.1,
        0.12,
        0.123,
        0.1234,
        10.1,
        10.12,
        10.123,
        10.1234,
        100.1,
        100.12,
        100.123,
        100.1234,
        1000.1,
        1000.12,
        1000.123,
        1000.1234,
        10000.1,
        10000.12,
        10000.123,
        10000.1234,
        100000.1,
        100000.12,
        100000.123,
        100000.1234,
        1000000.1,
        1000000.12,
        1000000.123,
        1000000.1234,
        10000000.1,
        10000000.12,
        10000000.123,
        10000000.1234,
        100000000.1,
        100000000.12,
        100000000.123,
        100000000.1234,
        1000000000.1,
        1000000000.12,
        1000000000.123,
        1000000000.1234,
        -0.1,
        -0.12,
        -0.123,
        -0.1234,
        -10.1,
        -10.12,
        -10.123,
        -10.1234,
        -100.1,
        -100.12,
        -100.123,
        -100.1234,
        -1000.1,
        -1000.12,
        -1000.123,
        -1000.1234,
        -10000.1,
        -10000.12,
        -10000.123,
        -10000.1234,
        -100000.1,
        -100000.12,
        -100000.123,
        -100000.1234,
        -1000000.1,
        -1000000.12,
        -1000000.123,
        -1000000.1234,
        -10000000.1,
        -10000000.12,
        -10000000.123,
        -10000000.1234,
        -100000000.1,
        -100000000.12,
        -100000000.123,
        -100000000.1234,
        -1000000000.1,
        -1000000000.12,
        -1000000000.123,
        -1000000000.1234,
        // Testing of special rounding cases
        1.9993,
        12.9993,
        123.9993,
        1234.9993,
        12345.9993,
        123456.9993,
        1234567.9993,
        12345678.9993,
        123456789.9993,
        1234567890.9993,
        1.9996,
        12.9996,
        123.9996,
        1234.9996,
        12345.9996,
        123456.9996,
        1234567.9996,
        12345678.9996,
        123456789.9996,
        1234567890.9996,
        -1.9993,
        -12.9993,
        -123.9993,
        -1234.9993,
        -12345.9993,
        -123456.9993,
        -1234567.9993,
        -12345678.9993,
        -123456789.9993,
        -1234567890.9993,
        -1.9996,
        -12.9996,
        -123.9996,
        -1234.9996,
        -12345.9996,
        -123456.9996,
        -1234567.9996,
        -12345678.9996,
        -123456789.9996,
        -1234567890.9996,
        109.9996,
        1099.9996,
        10999.9996,
        109999.9996,
        1099999.9996,
        10999999.9996,
        109999999.9996,
        1099999999.9996,
        -109.9996,
        -1099.9996,
        -10999.9996,
        -109999.9996,
        -1099999.9996,
        -10999999.9996,
        -109999999.9996,
        -1099999999.9996,
        1.9996,
        19.9996,
        199.9996,
        1999.9996,
        19999.9996,
        199999.9996,
        1999999.9996,
        19999999.9996,
        199999999.9996,
        1999999999.9996,
        -1.9996,
        -19.9996,
        -199.9996,
        -1999.9996,
        -19999.9996,
        -199999.9996,
        -1999999.9996,
        -19999999.9996,
        -199999999.9996,
        -1999999999.9996,
        // Testing for all nines cases
        9.9996,
        99.9996,
        999.9996,
        9999.9996,
        99999.9996,
        999999.9996,
        9999999.9996,
        99999999.9996,
        999999999.9996,
        9.999,
        99.999,
        999.999,
        9999.999,
        99999.999,
        999999.999,
        9999999.999,
        99999999.999,
        999999999.999,
        -9.9996,
        -99.9996,
        -999.9996,
        -9999.9996,
        -99999.9996,
        -999999.9996,
        -9999999.9996,
        -99999999.9996,
        -999999999.9996,
        -9.999,
        -99.999,
        -999.999,
        -9999.999,
        -99999.999,
        -999999.999,
        -9999999.999,
        -99999999.999,
        -999999999.999,
        // Testing for no Fractional part cases
        1.0,
        12.0,
        123.0,
        1234.0,
        12345.0,
        123456.0,
        1234567.0,
        12345678.0,
        123456789.0,
        1234567890.0,
        -1.0,
        -12.0,
        -123.0,
        -1234.0,
        -12345.0,
        -123456.0,
        -1234567.0,
        -12345678.0,
        -123456789.0,
        -1234567890.0,
        // Testing of tricky cases
        -2599.399999990123,
        -2599.339999990123,
        -2599.333999990123,
        1.000099999999818,
        1.000199999999818,
        1.000299999999818,
        1.000399999999818,
        1.000499999999818,
        1.000599999999818,
        1.000699999999818,
        1.000799999999818,
        1.000899999999818,
        1.000999999999818,
        1.2224999999999980,
        1.2224999999999981,
        1.2224999999999982,
        1.2224999999999983,
        1.2224999999999984,
        1.2224999999999985,
        1.2224999999999986,
        1.2224999999999987,
        1.2224999999999988,
        1.2224999999999989,
        1.2224999999999990,
        1.2224999999999991,
        1.2224999999999992,
        1.2224999999999993,
        1.2224999999999994,
        1.2224999999999995,
        1.2224999999999996,
        1.2224999999999997,
        1.2224999999999998,
        // 1.2225 and 1.2224999999999999 have the same double approximation
        1.2225,
        1.2225000000000001,
        1.2225000000000002,
        1.2225000000000003,
        1.2225000000000004,
        1.2225000000000005,
        1.2225000000000006,
        1.2225000000000007,
        1.2225000000000008,
        1.2225000000000009,
        1.2225000000000010,
        1.2225000000000011,
        1.2225000000000012,
        1.2225000000000013,
        1.2225000000000014,
        1.2225000000000015,
        1.2225000000000016,
        1.2225000000000017,
        1.2225000000000018,
        1.2225000000000019,
        // Tricky rounding cases around tie values
        100913.67050000005,
        199999.99895901306,
        251846.3465,
        253243.8825000001,
        365045.85349999997,
        314734.9615,
        541133.9755,
        858372.1225,
        1000999.9995000001,
        1347505.7825,
        3358844.1975,
        9997979.4085,
        9993743.1585,
        9938671.9085,
        3385302.5465,
        3404642.6605,
        3431280.0865,
        3438756.4754999997,
        3446053.7874999996,
        3457917.5125,
        3465393.9014999997,
        3484734.0154999997,
        3492031.3274999997,
        3503895.0525,
        3511371.4414999997,
        3518668.7534999996,
        3530532.4785,
        3538008.8674999997,
        3545306.1794999996,
        3557169.9045,
        3557348.9814999998,
        3564646.2934999997,
        3583986.4074999997,
        3591283.7194999997,
        3603147.4445,
        3610623.8334999997,
        3617921.1454999996,
        3629784.8705,
        3637261.2594999997,
        3656422.2965,
        3656601.3734999998,
        3663898.6854999997,
        3675762.4105,
        3683238.7994999997,
        3690536.1114999996,
        3702399.8365,
        3709876.2254999997,
        3717173.5374999996,
        3729037.2625,
        3736513.6514999997,
        3755853.7654999997,
        3763151.0774999997,
        3775014.8025,
        3782491.1914999997,
        3789788.5034999996,
        3801652.2285,
        3809128.6174999997,
        3816425.9294999996,
        3828289.6545,
        3828468.7314999998,
        3835766.0434999997,
        3855106.1574999997,
        3862403.4694999997,
        3874267.1945,
        3881743.5834999997,
        3889040.8954999996,
        3900904.6205,
        3908381.0094999997,
        3927542.0465,
        3927721.1234999998,
        3935018.4354999997,
        3946882.1605,
        3954358.5494999997,
        3961655.8614999996,
        3973519.5865,
        3980995.9754999997,
        3988293.2874999996,
        4000157.0125,
        4007633.4014999997,
        4026973.5154999997,
        4034270.8274999997,
        4046134.5525,
        4053610.9414999997,
        4060908.2534999996,
        4072771.9785,
        4080248.3674999997,
        4087545.6794999996,
        4099409.4045,
        4099588.4814999998,
        4106885.7934999997,
        4126225.9074999997,
        4133523.2194999997,
        4145386.9445,
        4152863.3334999997,
        4160160.6454999996,
        4172024.3705,
        4179500.7594999997,
        4198661.7965,
        4203407.2865,
        4210704.5985,
        4213435.4975
    };

    // --- The set of golden values for the currency formatting case --------
    static final double[] CurrencyGoldenValues = {
        // Testing of specific values
        +0.0,
        -0.0,
        Double.MIN_VALUE,
        Double.MIN_NORMAL,
        PROPERTY_CHECK_NEGATIVE_VALUE,
        PROPERTY_CHECK_POSITIVE_VALUE,
        -2147483647.996,
        2147483647.996,
        -1999999999.9949997,
        1999999999.9950003,
        // Testing fast-path range checks (all outside bounds)
        Double.NaN,
        Double.POSITIVE_INFINITY,
        Double.NEGATIVE_INFINITY,
        Double.MAX_VALUE,
        -9876543210.9876543,
        9876543210.9876543,
        -1234567890.1234567E128,
        1234567890.1234567E128,
        // Testing of integral string size
        1.12,
        12.12,
        123.12,
        1234.12,
        12345.12,
        123456.12,
        1234567.12,
        12345678.12,
        123456789.12,
        1234567890.12,
        -1.12,
        -12.12,
        -123.12,
        -1234.12,
        -12345.12,
        -123456.12,
        -1234567.12,
        -12345678.12,
        -123456789.12,
        -1234567890.12,
        // Testing of fractional string size
        0.1,
        0.12,
        0.123,
        10.1,
        10.12,
        10.123,
        100.1,
        100.12,
        100.123,
        1000.1,
        1000.12,
        1000.123,
        10000.1,
        10000.12,
        10000.123,
        100000.1,
        100000.12,
        100000.123,
        1000000.1,
        1000000.12,
        1000000.123,
        10000000.1,
        10000000.12,
        10000000.123,
        100000000.1,
        100000000.12,
        100000000.123,
        1000000000.1,
        1000000000.12,
        1000000000.123,
        -0.1,
        -0.12,
        -0.123,
        -10.1,
        -10.12,
        -10.123,
        -100.1,
        -100.12,
        -100.123,
        -1000.1,
        -1000.12,
        -1000.123,
        -10000.1,
        -10000.12,
        -10000.123,
        -100000.1,
        -100000.12,
        -100000.123,
        -1000000.1,
        -1000000.12,
        -1000000.123,
        -10000000.1,
        -10000000.12,
        -10000000.123,
        -100000000.1,
        -100000000.12,
        -100000000.123,
        -1000000000.1,
        -1000000000.12,
        -1000000000.123,
        // Testing of special rounding cases
        1.993,
        12.993,
        123.993,
        1234.993,
        12345.993,
        123456.993,
        1234567.993,
        12345678.993,
        123456789.993,
        1234567890.993,
        1.996,
        12.996,
        123.996,
        1234.996,
        12345.996,
        123456.996,
        1234567.996,
        12345678.996,
        123456789.996,
        1234567890.996,
        -1.993,
        -12.993,
        -123.993,
        -1234.993,
        -12345.993,
        -123456.993,
        -1234567.993,
        -12345678.993,
        -123456789.993,
        -1234567890.993,
        -1.996,
        -12.996,
        -123.996,
        -1234.996,
        -12345.996,
        -123456.996,
        -1234567.996,
        -12345678.996,
        -123456789.996,
        -1234567890.996,
        109.996,
        1099.996,
        10999.996,
        109999.996,
        1099999.996,
        10999999.996,
        109999999.996,
        1099999999.996,
        -109.996,
        -1099.996,
        -10999.996,
        -109999.996,
        -1099999.996,
        -10999999.996,
        -109999999.996,
        -1099999999.996,
        1.996,
        19.996,
        199.996,
        1999.996,
        19999.996,
        199999.996,
        1999999.996,
        19999999.996,
        199999999.996,
        1999999999.996,
        -1.996,
        -19.996,
        -199.996,
        -1999.996,
        -19999.996,
        -199999.996,
        -1999999.996,
        -19999999.996,
        -199999999.996,
        -1999999999.996,
        // Testing of all nines cases
        9.996,
        99.996,
        999.996,
        9999.996,
        99999.996,
        999999.996,
        9999999.996,
        99999999.996,
        999999999.996,
        9.99,
        99.99,
        999.99,
        9999.99,
        99999.99,
        999999.99,
        9999999.99,
        99999999.99,
        999999999.99,
        -9.996,
        -99.996,
        -999.996,
        -9999.996,
        -99999.996,
        -999999.996,
        -9999999.996,
        -99999999.996,
        -999999999.996,
        -9.99,
        -99.99,
        -999.99,
        -9999.99,
        -99999.99,
        -999999.99,
        -9999999.99,
        -99999999.99,
        -999999999.99,
        // Testing of no Fractional part cases
        1.0,
        12.0,
        123.0,
        1234.0,
        12345.0,
        123456.0,
        1234567.0,
        12345678.0,
        123456789.0,
        1234567890.0,
        -1.0,
        -12.0,
        -123.0,
        -1234.0,
        -12345.0,
        -123456.0,
        -1234567.0,
        -12345678.0,
        -123456789.0,
        -1234567890.0,
        // Testing of tricky cases
        -2599.399999990123,
        -2599.339999990123,
        -2599.333999990123,
        1.000999999999818,
        1.001999999999818,
        1.002999999999818,
        1.003999999999818,
        1.004999999999818,
        1.005999999999818,
        1.006999999999818,
        1.007999999999818,
        1.008999999999818,
        1.009999999999818,
        1.224999999999980,
        1.224999999999981,
        1.224999999999982,
        1.224999999999983,
        1.224999999999984,
        1.224999999999985,
        1.224999999999986,
        1.224999999999987,
        1.224999999999988,
        1.224999999999989,
        1.224999999999990,
        1.224999999999991,
        1.224999999999992,
        1.224999999999993,
        1.224999999999994,
        1.224999999999995,
        1.224999999999996,
        1.224999999999997,
        1.224999999999998,
        1.224999999999999,
        1.225,
        1.225000000000001,
        1.225000000000002,
        1.225000000000003,
        1.225000000000004,
        1.225000000000005,
        1.225000000000006,
        1.225000000000007,
        1.225000000000008,
        1.225000000000009,
        1.225000000000010,
        1.225000000000011,
        1.225000000000012,
        1.225000000000013,
        1.225000000000014,
        1.225000000000015,
        1.225000000000016,
        1.225000000000017,
        1.225000000000018,
        1.225000000000019,
        // Tricky rounding cases around tie values
        1009136.7050000005,
        2518463.465,
        2532438.825000001,
        3650458.5349999997,
        3147349.615,
        5411339.755,
        8583721.225,
        13475057.825,
        33588441.975,
        99979794.085,
        99937431.585,
        99386719.085,
        33853025.465,
        34046426.605,
        34312800.865,
        34387564.754999997,
        34460537.874999996,
        34579175.125,
        34653939.014999997,
        34847340.154999997,
        34920313.274999997,
        35038950.525,
        35113714.414999997,
        35186687.534999996,
        35305324.785,
        35380088.674999997,
        35453061.794999996,
        35571699.045,
        35573489.814999998,
        35646462.934999997,
        35839864.074999997,
        35912837.194999997,
        36031474.445,
        36106238.334999997,
        36179211.454999996,
        36297848.705,
        36372612.594999997,
        36564222.965,
        36566013.734999998,
        36638986.854999997,
        36757624.105,
        36832387.994999997,
        36905361.114999996,
        37023998.365,
        37098762.254999997,
        37171735.374999996,
        37290372.625,
        37365136.514999997,
        37558537.654999997,
        37631510.774999997,
        37750148.025,
        37824911.914999997,
        37897885.034999996,
        38016522.285,
        38091286.174999997,
        38164259.294999996,
        38282896.545,
        38284687.314999998,
        38357660.434999997,
        38551061.574999997,
        38624034.694999997,
        38742671.945,
        38817435.834999997,
        38890408.954999996,
        39009046.205,
        39083810.094999997,
        39275420.465,
        39277211.234999998,
        39350184.354999997,
        39468821.605,
        39543585.494999997,
        39616558.614999996,
        39735195.865,
        39809959.754999997,
        39882932.874999996,
        40001570.125,
        40076334.014999997,
        40269735.154999997,
        40342708.274999997,
        40461345.525,
        40536109.414999997,
        40609082.534999996,
        40727719.785,
        40802483.674999997,
        40875456.794999996,
        40994094.045,
        40995884.814999998,
        41068857.934999997,
        41262259.074999997,
        41335232.194999997,
        41453869.445,
        41528633.334999997,
        41601606.454999996,
        41720243.705,
        41795007.594999997,
        41986617.965,
        42034072.865,
        42107045.985,
        42134354.975
    };
}
